<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\web\UploadedFile;
use yii\web\NotFoundHttpException;
use app\models\EmployeeProfile;
use app\models\User;

class ProfileController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return !Yii::$app->user->identity->isAdmin();
                        }
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    return $this->redirect(['/site/index']);
                }
            ],
        ];
    }

    /**
     * Главная страница личного кабинета
     */
    public function actionIndex()
    {
        $user = Yii::$app->user->identity;
        $profile = EmployeeProfile::findOne(['user_id' => $user->id]);

        return $this->render('index', [
            'user' => $user,
            'profile' => $profile,
        ]);
    }

    /**
     * Создание анкеты (многошаговая форма)
     */
    public function actionCreate($step = 1)
    {
        $user = Yii::$app->user->identity;
        
        // Проверяем, есть ли уже анкета
        $existingProfile = EmployeeProfile::findOne(['user_id' => $user->id]);
        if ($existingProfile) {
            Yii::$app->session->setFlash('info', 'Анкета уже заполнена. Вы можете ее редактировать.');
            return $this->redirect(['update']);
        }

        $model = new EmployeeProfile();
        $model->user_id = $user->id;

        return $this->render('create', [
            'model' => $model,
            'user' => $user,
            'step' => $step,
        ]);
    }

    /**
     * Обработка отправки формы анкеты
     */
    public function actionProcess()
    {
        $user = Yii::$app->user->identity;
        
        if (Yii::$app->request->isPost) {
            $step = Yii::$app->request->post('step', 1);
            $model = new EmployeeProfile();
            
            // Загружаем данные из сессии
            $session = Yii::$app->session;
            $formData = $session->get('formData', []);
            
            // Добавляем новые данные
            $newData = Yii::$app->request->post('EmployeeProfile', []);
            $formData = array_merge($formData, $newData);
            $session->set('formData', $formData);
            
            // Если это последний шаг, сохраняем анкету
            if ($step == 4) {
                $model->attributes = $formData;
                $model->user_id = $user->id;
                $model->status = EmployeeProfile::STATUS_NEW;
                
                // Загрузка файлов
                $model->photoFile = UploadedFile::getInstance($model, 'photoFile');
                $model->passportScanFile = UploadedFile::getInstance($model, 'passportScanFile');
                $model->diplomaScanFile = UploadedFile::getInstance($model, 'diplomaScanFile');
                $model->snilsScanFile = UploadedFile::getInstance($model, 'snilsScanFile');
                
                if ($model->save()) {
                    if ($model->upload()) {
                        Yii::$app->session->setFlash('success', 
                            '✅ Анкета успешно создана и отправлена на проверку!'
                        );
                        $session->remove('formData'); // Очищаем сессию
                    } else {
                        Yii::$app->session->setFlash('warning', 
                            'Анкета сохранена, но возникли проблемы с загрузкой файлов.'
                        );
                    }
                    return $this->redirect(['index']);
                } else {
                    Yii::$app->session->setFlash('error', 
                        'Ошибка при сохранении анкеты. Проверьте введенные данные.'
                    );
                    return $this->redirect(['create', 'step' => 4]);
                }
            }
            
            // Переходим к следующему шагу
            return $this->redirect(['create', 'step' => $step + 1]);
        }
        
        return $this->redirect(['create']);
    }

    /**
     * Редактирование анкеты
     */
    public function actionUpdate()
    {
        $user = Yii::$app->user->identity;
        $model = EmployeeProfile::findOne(['user_id' => $user->id]);

        if (!$model) {
            Yii::$app->session->setFlash('warning', 'Сначала создайте анкету.');
            return $this->redirect(['create']);
        }

        // Если анкета уже проверена администратором, запрещаем редактирование
        if ($model->status === EmployeeProfile::STATUS_ACCEPTED) {
            Yii::$app->session->setFlash('info', 
                'Анкета уже принята администратором. Для внесения изменений обратитесь к администратору.'
            );
            return $this->redirect(['index']);
        }

        if ($model->load(Yii::$app->request->post())) {
            $model->photoFile = UploadedFile::getInstance($model, 'photoFile');
            $model->passportScanFile = UploadedFile::getInstance($model, 'passportScanFile');
            $model->diplomaScanFile = UploadedFile::getInstance($model, 'diplomaScanFile');
            $model->snilsScanFile = UploadedFile::getInstance($model, 'snilsScanFile');

            // При редактировании сбрасываем статус на "новая"
            if ($model->status !== EmployeeProfile::STATUS_ACCEPTED) {
                $model->status = EmployeeProfile::STATUS_NEW;
                $model->reviewed_by = null;
                $model->reviewed_at = null;
                $model->review_comment = null;
            }

            if ($model->save()) {
                if ($model->upload()) {
                    Yii::$app->session->setFlash('success', 
                        '✅ Данные анкеты успешно обновлены и отправлены на повторную проверку!'
                    );
                } else {
                    Yii::$app->session->setFlash('warning', 
                        'Данные обновлены, но возникли проблемы с загрузкой файлов.'
                    );
                }
                return $this->redirect(['index']);
            } else {
                Yii::$app->session->setFlash('error', 
                    'Ошибка при обновлении анкеты. Проверьте введенные данные.'
                );
            }
        }

        return $this->render('update', [
            'model' => $model,
            'user' => $user,
        ]);
    }

    /**
     * Просмотр анкеты
     */
    public function actionView()
    {
        $user = Yii::$app->user->identity;
        $model = EmployeeProfile::findOne(['user_id' => $user->id]);

        if (!$model) {
            Yii::$app->session->setFlash('warning', 'Сначала создайте анкету.');
            return $this->redirect(['create']);
        }

        return $this->render('view', [
            'model' => $model,
            'user' => $user,
        ]);
    }
}